package com.example.sefinsa_app;


import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.sqlite.SQLiteDatabase;
import android.net.ConnectivityManager;
import android.net.Network;
import android.net.NetworkCapabilities;
import android.os.Build;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import androidx.annotation.NonNull;
import androidx.annotation.RequiresApi;
import androidx.work.BackoffPolicy;
import androidx.work.Constraints;
import androidx.work.ExistingPeriodicWorkPolicy;
import androidx.work.ExistingWorkPolicy;
import androidx.work.ListenableWorker;
import androidx.work.NetworkType;
import androidx.work.OneTimeWorkRequest;
import androidx.work.Worker;
import androidx.work.WorkerParameters;

import com.android.volley.DefaultRetryPolicy;
import com.android.volley.Request;
import com.android.volley.VolleyError;
import com.android.volley.toolbox.JsonObjectRequest;
import com.example.sefinsa_app.api.API;
import com.example.sefinsa_app.migrations.DatabaseHelper;
import com.example.sefinsa_app.models.Cliente;
import com.google.gson.Gson;
import com.google.gson.JsonParseException;
import com.google.gson.JsonSyntaxException;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import androidx.work.PeriodicWorkRequest;
import androidx.work.WorkManager;

public class ClientesWorker extends Worker {
    private SharedPreferences sesion;
    private final ConnectivityManager connectivityManager;
    private final boolean isWorkEnqueued = false; // Bandera para evitar encolar múltiples veces
    public static final int BATCH_SIZE_CLIENTES = 999999; // Tamaño del lote
    private static final int MAX_RETRIES = 3; // Número máximo de reintentos
    private int retryCount = 0;
    public static int currentPageClientes = 0; // Página actual
    public static boolean allDataLoadedClientes = false;
    public static boolean isTaskClientesCompleted = false;
    public static final String CLIENTES_WORKER_TAG = "ClientesWorker";
    @RequiresApi(api = Build.VERSION_CODES.N)
    public ClientesWorker(Context context, WorkerParameters workerParams) {
        super(context, workerParams);
        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);
        connectivityManager = (ConnectivityManager)context.getSystemService(Context.CONNECTIVITY_SERVICE);
    }

    @RequiresApi(api = Build.VERSION_CODES.N)
    @NonNull
    @Override
    public ListenableWorker.Result doWork() {
        SQLiteDatabase db = null;
        try {
            if (!isNetworkAvailable(getApplicationContext())) {
                Log.d("ClientesWorker", "No internet connection, retrying later.");
                return retryOrFailure();
            }

            if (!isWithinAllowedTime()) {
                Log.d("ClientesWorker", "Current time is outside allowed range, retrying later.");
                return retryOrFailure();
            }

            DatabaseHelper dbHelper = new DatabaseHelper(getApplicationContext());
            db = dbHelper.getWritableDatabase();

            if (db == null || !db.isOpen()) {
                Log.e("ClientesWorker", "Fallo al abrir la base de datos Clientes.");
                return ListenableWorker.Result.failure();
            }

            if (!allDataLoadedClientes) {
                loadAllData(db);
                Log.d("ClientesWorker", "ENTRO EN CARGAR TODOS LOS DATOS........................");
            } else {
                checkForUpdates(getApplicationContext(), db);
                Log.d("ClientesWorker", "ENTRO EN ACTUALIZAR TODOS LOS DATOS.......................");
            }

            reprogramarWorker();
            return ListenableWorker.Result.success();

        } catch (Exception e) {
            Log.e("ClientesWorker", "Error in doWork", e);
            return ListenableWorker.Result.failure();
        } finally {
            if (isTaskClientesCompleted && db != null && db.isOpen()) {
                Log.d("ClientesWorker", "Base de datos cerrada desde worker CLIENTES.");
                db.close();
            }
        }
    }
    private void reprogramarWorker() {
        WorkManager.getInstance(getApplicationContext()).enqueueUniqueWork(
                "ClientesWorker",
                ExistingWorkPolicy.REPLACE, // Evita que se creen trabajos duplicados
                new OneTimeWorkRequest.Builder(ClientesWorker.class)
                        .setInitialDelay(14, TimeUnit.MINUTES)
                        .build()
        );

        Log.d("ClientesWorker", "Worker ClientesWorker reprogramado para ejecutarse en 5 minuto.");
    }
    private ListenableWorker.Result retryOrFailure() {
        if (retryCount < MAX_RETRIES) {
            retryCount++;
            return ListenableWorker.Result.retry();
        } else {
            return ListenableWorker.Result.failure();
        }
    }

    private boolean isWithinAllowedTime() {
        Calendar calendar = Calendar.getInstance();
        int hourOfDay = calendar.get(Calendar.HOUR_OF_DAY);
        return hourOfDay >= 7 && hourOfDay <= 22;
    }


    @RequiresApi(api = Build.VERSION_CODES.N)
    private void loadAllData(SQLiteDatabase db) {
        Handler mainHandler = new Handler(Looper.getMainLooper());

        Context context = getApplicationContext();
        sesion = context.getSharedPreferences("sesion", Context.MODE_PRIVATE);

        JSONObject data = new JSONObject();
        try {
            data.put("func", "indexApp");
            data.put("ruta_id", sesion.getString("rutas", ""));
        } catch (JSONException e) {
            e.printStackTrace();
            return;
        }
        Log.d("Clientes", "Parametros enviados WORKER CLIENTES indexApp: " + data);
        Log.d("1", "loadAllData: ");
        Log.d("asdasdasd", data.toString());
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlClientes, data,
                response -> {
                    Log.d("alkslkdas", String.valueOf(response));
                        try {
                            JSONArray data1 = response.getJSONArray("data");
                            List<Cliente> batchClientes = new ArrayList<>();
                            for (int i = 0; i < data1.length(); i++) {
                                JSONObject obj = data1.getJSONObject(i);

                                try {
                                    // Intentar la conversión de Gson
                                    Cliente cliente = new Gson().fromJson(obj.toString(), Cliente.class);
                                    batchClientes.add(cliente);
                                } catch (JsonSyntaxException e) {
                                    Log.e("ERROR_JSON", "Error de sintaxis JSON en el registro " + i + ": " + obj.toString(), e);
                                } catch (JsonParseException e) {
                                    Log.e("ERROR_JSON", "Error al analizar el JSON en el registro " + i + ": " + obj.toString(), e);
                                } catch (Exception e) {
                                    Log.e("ERROR_JSON", "Error inesperado en el registro " + i + ": " + obj.toString(), e);
                                }
                            }

                            insertClientesIntoSQLiteAllDatos(context, batchClientes, db);

                            if (batchClientes.size() < BATCH_SIZE_CLIENTES) {
                                allDataLoadedClientes = true;
                            }
                        } catch (JSONException e) {
                            e.printStackTrace();
                        }
                },
                error -> mainHandler.post(() -> handleNetworkError(context, error, "loadAllData"))
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000,
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));

        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }

    private void insertClientesIntoSQLiteAllDatos(Context context, List<Cliente> clientes, SQLiteDatabase db) {
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }
        Log.d("SQLite", "Insertando clientes en lote...");
        db.beginTransaction(); // Iniciar la transacción
        try {
            for (Cliente cliente : clientes) {
                ContentValues values = new ContentValues();
                values.put("id", cliente.getId());
                values.put("nombre_completo", cliente.getNombre_completo() != null ? cliente.getNombre_completo() : "");
                values.put("direccion", cliente.getDireccion() != null ? cliente.getDireccion() : "");
                values.put("telefono", cliente.getTelefono() != null ? cliente.getTelefono() : "");
                values.put("garantias", cliente.getGarantias() != null ? cliente.getGarantias() : "");
                values.put("carpeta_comprobantes", cliente.getCarpeta_comprobantes() != null ? cliente.getCarpeta_comprobantes() : "");
                values.put("carpeta_garantias", cliente.getCarpeta_garantias() != null ? cliente.getCarpeta_garantias() : "");
                values.put("otras_referencias", cliente.getOtras_referencias() != null ? cliente.getOtras_referencias() : "");
                values.put("ruta_id", cliente.getRuta_id() != null ? cliente.getRuta_id() : "NULLOO");
                values.put("poblacion_id", cliente.getPoblacion_id() != null ? cliente.getPoblacion_id() : "NULLOO");
                values.put("colocadora_id", cliente.getColocadora_id() != null ? cliente.getColocadora_id() : "NULLOO");
                values.put("created_at", cliente.getCreated_at() != null ? cliente.getCreated_at() : "");
                values.put("updated_at", cliente.getUpdated_at() != null ? cliente.getUpdated_at() : "");
                values.put("update_comprobantes", cliente.getUpdate_comprobantes_cliente() != null ? cliente.getUpdate_comprobantes_cliente() : "");
                values.put("update_garantias", cliente.getUpdate_garantias_cliente() != null ? cliente.getUpdate_garantias_cliente() : "");
                values.put("prestanombre", cliente.getPrestanombre() != null ? cliente.getPrestanombre() : "");
                values.put("aval_id", cliente.getAval_id() != null ? cliente.getAval_id() : "");

                db.insert("clientes", null, values);
            }

            // Marcar la transacción como exitosa después de insertar todos los datos
            db.setTransactionSuccessful();
            Log.d("SQLite", "Todos los clientes insertados correctamente.");
        } catch (Exception e) {
            Log.e("SQLite", "Error al insertar clientes en lote: " + e.getMessage(), e);
        } finally {
            db.endTransaction(); // Finaliza la transacción
            // No cierres la base de datos aquí
            isTaskClientesCompleted = true;
        }
    }

    private void handleNetworkError(Context context, VolleyError error, String from) {
        String errorMessage = error.getMessage();
        Log.e("ClientesWorker", "Error occurred in " + from + ": " + errorMessage, error);

        if (error.networkResponse != null) {
            Log.e("ClientesWorker", "Código de estado: " + error.networkResponse.statusCode);
            try {
                String responseBody = new String(error.networkResponse.data, "UTF-8");
                Log.e("ClientesWorker", "Response body: " + responseBody);
            } catch (Exception e) {
                Log.e("ClientesWorker", "Error reading response body", e);
            }
        }
    }

    private void checkForUpdates(Context context, SQLiteDatabase db) {
        Handler mainHandler = new Handler(Looper.getMainLooper());
        Gson gson = new Gson();

        DatabaseHelper dbHelper = new DatabaseHelper(context);

        // Obtener los últimos 40 pagos desde SQLite
        List<Cliente> localClientes = dbHelper.getLast30ClientesFromSQLite();

        // Crear el objeto JSON para los parámetros de la solicitud
        JSONObject data = new JSONObject();
        try {
            data.put("func", "indexApp_Upd");
            data.put("ruta_id", sesion.getString("rutas", ""));
            //Log.d("CLIENTES", "Parametros enviados WORKER indexApp_Upd: " + data.toString());
        } catch (JSONException e) {
            e.printStackTrace();
            return; // Salir si hay un error al crear el JSON
        }

        if (context != null) {
            Log.d("2", "loadAllData: ");
                JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlClientes, data,
                        response -> {
                                try {
                                    JSONArray data1 = response.getJSONArray("data");
                                    List<Cliente> newClientes = new ArrayList<>();

                                    for (int i = 0; i < data1.length(); i++) {
                                        JSONObject obj = data1.getJSONObject(i);
                                        Cliente serverData = gson.fromJson(obj.toString(), Cliente.class);
                                        newClientes.add(serverData);
                                    }
                                    updateClienteInSQLite(context, newClientes, localClientes, db);
                                } catch (JSONException e) {
                                    e.printStackTrace();
                                }
                        },
                        error -> {
                            mainHandler.post(() -> {
                                if (context != null) {
                                    Log.e("ClientesWorker", "Error occurred: " + error.getMessage(), error);
                                } else {
                                    Log.e("ClientesWorker", "Context is null, cannot handle network error.");
                                }
                            });
                        }
                );
                request.setRetryPolicy(new DefaultRetryPolicy(
                        10000,
                        DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                        DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
                VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
        }
    }

    private void insertClientesIntoSQLite(Context context, List<Cliente> clientes, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        Log.d("SQLite", "Insertando clientes en lote...");
        // Iniciar una transacción para insertar en lote

        try {
            for (Cliente cliente : clientes) {
                ContentValues values = new ContentValues();
                values.put("id", cliente.getId());
                values.put("nombre_completo", cliente.getNombre_completo() != null ? cliente.getNombre_completo() : "");
                values.put("direccion", cliente.getDireccion() != null ? cliente.getDireccion() : "");
                values.put("telefono", cliente.getTelefono() != null ? cliente.getTelefono() : "");
                values.put("garantias", cliente.getGarantias() != null ? cliente.getGarantias() : "");
                values.put("carpeta_comprobantes", cliente.getCarpeta_comprobantes() != null ? cliente.getCarpeta_comprobantes() : "");
                values.put("carpeta_garantias", cliente.getCarpeta_garantias() != null ? cliente.getCarpeta_garantias() : "");
                values.put("otras_referencias", cliente.getOtras_referencias() != null ? cliente.getOtras_referencias() : "");
                values.put("ruta_id", cliente.getRuta_id() != null ? cliente.getRuta_id() : "NULLOO");
                values.put("poblacion_id", cliente.getPoblacion_id() != null ? cliente.getPoblacion_id() : "NULLOO");
                values.put("colocadora_id", cliente.getColocadora_id() != null ? cliente.getColocadora_id() : "NULLOO");
                values.put("created_at", cliente.getCreated_at() != null ? cliente.getCreated_at() : "");
                values.put("updated_at", cliente.getUpdated_at() != null ? cliente.getUpdated_at() : "");
                values.put("update_comprobantes", cliente.getUpdate_comprobantes_cliente() != null ? cliente.getUpdate_comprobantes_cliente() : "");
                values.put("update_garantias", cliente.getUpdate_garantias_cliente() != null ? cliente.getUpdate_garantias_cliente() : "");
                values.put("prestanombre", cliente.getPrestanombre() != null ? cliente.getPrestanombre() : "");
                values.put("aval_id", cliente.getAval_id() != null ? cliente.getAval_id() : "");

                // Insertar el cliente y verificar el resultado
                long rowId = db.insertWithOnConflict("clientes", null, values, SQLiteDatabase.CONFLICT_REPLACE);
                if (rowId != -1) {
                    //Log.d("SQLite", "Cliente insertado correctamente: ID = " + cliente.getId());
                    remainingComparisons.decrementAndGet();
                } else {
                    //Log.e("SQLite", "Error al insertar el cliente: ID = " + cliente.getId());
                    remainingComparisons.decrementAndGet();
                }
            }

            //Log.d("SQLite", "Todos los clientes han sido insertados correctamente:" + clientes.size());

        } catch (Exception e) {
            Log.e("SQLite", "Error al insertar clientes en lote: " + e.getMessage(), e);
        }
    }

    private void updateClienteInSQLite(Context context, List<Cliente> serverData, List<Cliente> localClientes, SQLiteDatabase db) {
        if (!db.isOpen()) {
            DatabaseHelper dbHelper = new DatabaseHelper(context);
            db = dbHelper.getWritableDatabase();
        }

        // Crear un conjunto para buscar rápidamente si un ID existe en la lista local
        Set<String> localClienteIds = new HashSet<>();
        for (Cliente localCliente : localClientes) {
            localClienteIds.add(localCliente.getId());
        }
        AtomicInteger remainingComparisons = new AtomicInteger(serverData.size());

        // Sincronizar clientes desde el servidor a la base de datos local
        for (Cliente serverCliente : serverData) {
            if (localClienteIds.contains(serverCliente.getId())) {
                // Si el cliente existe en local, actualizamos los valores si es necesario
                Cliente localCliente = null;
                for (Cliente cliente : localClientes) {
                    if (cliente.getId().equals(serverCliente.getId())) {
                        localCliente = cliente;
                        break;
                    }
                }

                if (localCliente != null) {
                    String localUpdatedAt = localCliente.getUpdated_at();

                    if (localUpdatedAt == null || localUpdatedAt.isEmpty()) {
                        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.O) {
                            LocalDateTime now = LocalDateTime.now(); // Hora actual
                            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = now.format(formatter);
                        } else {
                            // Manejo para versiones antiguas (opcional, pero recomendable si soportas < API 26)
                            SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX"));
                            localUpdatedAt = sdf.format(new Date()); // Hora actual sin modificaciones
                        }
                        // Insertar la hora actual en la base de datos SQLite
                        ContentValues values = new ContentValues();
                        values.put("updated_at", localUpdatedAt);

                        int rowsUpdated = db.update("clientes", values, "id = ?", new String[]{localCliente.getId()});

                        if (rowsUpdated > 0) {
                            Log.d("SQLite", "Updated 'updated_at' for Clientes ID: " + localCliente.getId());
                        } else {
                            Log.e("SQLite", "Failed to update 'updated_at' for Clientes ID: " + localCliente.getId());
                        }
                    }
                    if (localUpdatedAt == null || localUpdatedAt.isEmpty() || isNewer(serverCliente.getUpdated_at(), localUpdatedAt)) {
                        // El servidor tiene una versión más nueva del cliente
                        updateValuesInDatabase(db, serverCliente, remainingComparisons);
                    } else if (isNewer(localUpdatedAt, serverCliente.getUpdated_at())) {
                        // El registro local está más actualizado, enviarlo al servidor
                        enviarDatosLocalAlServidor(context, localCliente, db, remainingComparisons);
                    } else {
                        // El registros son iguales, no hacer nada
                        remainingComparisons.decrementAndGet();
                    }
                }
            } else {
                // Si no existe en local, insertarlo
                insertClientesIntoSQLite(context, Collections.singletonList(serverCliente), db, remainingComparisons);
            }
        }
        // Verificar si es hora de cerrar la base de datos
        if (remainingComparisons.get() == 0) {
            isTaskClientesCompleted = true;
        }
    }

    private void enviarDatosLocalAlServidor(Context context, Cliente clienteLocal, SQLiteDatabase db, AtomicInteger remainingComparisons) {
        Handler mainHandler = new Handler(Looper.getMainLooper());
        remainingComparisons.decrementAndGet(); // Reducir el contador después de recibir la respuesta
        if (context == null) {
            Log.e("ClientesWorker", "Context is null. Cannot send data to server.");
            remainingComparisons.decrementAndGet(); // Reducir el contador si no se puede enviar
            return;
        }

        JSONObject data = new JSONObject();
        try {
            data.put("func", "updateFromLocal");
            data.put("id", clienteLocal.getId());
            data.put("nombre", clienteLocal.getNombre_completo());
            data.put("direccion", clienteLocal.getDireccion());
            data.put("telefono", clienteLocal.getTelefono());
            data.put("garantias", clienteLocal.getGarantias());
            data.put("otras_referencias", clienteLocal.getOtras_referencias());
            data.put("ruta_id", clienteLocal.getRuta_cliente());
            data.put("poblacion_id", clienteLocal.getPoblacion_cliente());
            data.put("colocadora_id", clienteLocal.getColocadora_id());
            data.put("created_at", clienteLocal.getCreated_at());
            data.put("updated_at", clienteLocal.getUpdated_at());
            data.put("prestanombre", clienteLocal.getPrestanombre());
            data.put("aval_id", clienteLocal.getAval_id());
        } catch (JSONException e) {
            Log.e("ClientesWorker", "Error creating JSON for enviarDatosLocalAlServidor", e);
            remainingComparisons.decrementAndGet(); // Reducir el contador si hay un error
            return;
        }

        Log.d("LocalAlServidor", "ClientesWorker enviarDatosLocalAlServidor Datos enviados: " + data.toString());

        Log.d("3", "loadAllData: ");
        JsonObjectRequest request = new JsonObjectRequest(Request.Method.POST, API.urlClientes, data,
                response -> {
                    Log.d("ClientesWorker", "Datos locales enviados y respuesta recibida: " + response.toString());
                },
                error -> {
                    Log.e("ClientesWorker", "Error al enviar datos al servidor", error);
                }
        );

        request.setRetryPolicy(new DefaultRetryPolicy(
                10000, // 10 segundos de timeout
                DefaultRetryPolicy.DEFAULT_MAX_RETRIES,
                DefaultRetryPolicy.DEFAULT_BACKOFF_MULT));
        VolleySingleton.getInstance(context.getApplicationContext()).addToRequestQueue(request);
    }


    private void updateValuesInDatabase(SQLiteDatabase db, Cliente cliente, AtomicInteger remainingComparisons) {
        ContentValues values = new ContentValues();
        values.put("nombre_completo", cliente.getNombre_completo());
        values.put("direccion", cliente.getDireccion());
        values.put("telefono", cliente.getTelefono());
        values.put("garantias", cliente.getGarantias());
        values.put("carpeta_comprobantes", cliente.getCarpeta_comprobantes());
        values.put("carpeta_garantias", cliente.getCarpeta_garantias());
        values.put("otras_referencias", cliente.getOtras_referencias());
        values.put("ruta_id", cliente.getRuta_id() != null ? cliente.getRuta_id() : "0");
        values.put("poblacion_id", cliente.getPoblacion_id() != null ? cliente.getPoblacion_id() : "0");
        values.put("colocadora_id", cliente.getColocadora_id() != null ? cliente.getColocadora_id() : "0");
        values.put("created_at", cliente.getCreated_at());
        values.put("updated_at", cliente.getUpdated_at());
        values.put("update_comprobantes", cliente.getUpdate_comprobantes_cliente());
        values.put("update_garantias", cliente.getUpdate_garantias_cliente());
        values.put("aval_id", cliente.getAval_id());
        values.put("prestanombre", cliente.getPrestanombre() != null ? cliente.getPrestanombre() : "0");
        int rowsUpdated = db.update("clientes", values, "id=?", new String[]{String.valueOf(cliente.getId())});

        if (rowsUpdated > 0) {
            Log.d("SQLite", "Registro actualizado correctamente Cliente: ID = " + cliente.getId());
            remainingComparisons.decrementAndGet();
        } else {
            Log.e("SQLite", "Error al actualizar el registro. ID no encontrado o fallo en la base de datos.");
            remainingComparisons.decrementAndGet();
        }
        // Reducir el contador después de completar la operación

    }

    private boolean isNewer(String serverDate, String localDate) {
        if (serverDate == null || localDate == null || serverDate.isEmpty() || localDate.isEmpty()) {
            Log.e("isNewer", "Una de las fechas es nula o vacía: serverDate=" + serverDate + ", localDate=" + localDate);
            return false; // O decide cómo manejar este caso
        }

        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss", new Locale("es", "MX")); // Ajusta el formato según tu necesidad
        try {
            Date server = sdf.parse(serverDate);
            Date local = sdf.parse(localDate);

            return server != null && local != null && server.after(local);
        } catch (ParseException e) {
            Log.e("isNewer", "Error parsing dates", e);
            return false; // En caso de error, asumir que no es más reciente
        }
    }



    private boolean isNetworkAvailable(Context context) {
        if (connectivityManager == null) {return false;
        }
        Network network = null;
        if (android.os.Build.VERSION.SDK_INT >= android.os.Build.VERSION_CODES.M) {
            network = connectivityManager.getActiveNetwork();
        }
        if (network == null) {
            return false;
        }
        NetworkCapabilities networkCapabilities = connectivityManager.getNetworkCapabilities(network);
        return networkCapabilities != null &&
                (networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_WIFI) ||
                        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_CELLULAR) ||
                        networkCapabilities.hasTransport(NetworkCapabilities.TRANSPORT_ETHERNET));
    }


    public static void enqueueWork(Context context) {
        // Configurar restricciones, como conexión a Internet
        Constraints constraints = new Constraints.Builder()
                .setRequiredNetworkType(NetworkType.CONNECTED)
                .build();

        // Crear un OneTimeWorkRequest (trabajo único)
        OneTimeWorkRequest oneTimeWorkRequest = new OneTimeWorkRequest.Builder(ClientesWorker.class)
                .setConstraints(constraints)
                //.setBackoffCriteria(BackoffPolicy.EXPONENTIAL, 10, TimeUnit.MINUTES) // Retraso para reintentos en caso de fallo
                .addTag(CLIENTES_WORKER_TAG) // Agregar una etiqueta para identificar el trabajo
                .build();

        // Encolar el trabajo único
        WorkManager.getInstance(context.getApplicationContext()).enqueueUniqueWork(
                "ClientesWorker", // Identificador único del trabajo
                ExistingWorkPolicy.REPLACE, // Reemplazar si existe uno previo con el mismo ID
                oneTimeWorkRequest
        );
    }
}
